'''
ezffmpeg.py
動画変換限定のffmpegフロントエンド

できること
・動画の時間切り出し
・映像、音声、字幕ストリームの選択（コンテナによる）
・映像コーデック変更（コンテナによる）
・映像の解像度変更（アスペクト比維持）
・映像の選択領域の切り抜き
・音声コーデック変更（コンテナによる）
・音声ボリュームの増減
・extcodecs.pyでコンテナ別の主な対応コーデックを定義しているが
  コンボボックスには任意のコーデックを入力可能

今後の予定など
・アスペクト比を維持しない解像度の変更
  （歪んだ映像を元に戻す使い方もあると思うので）
・映像、音声、字幕データの抽出

---------------------------------------------
履歴
R5.7.10
  動作確認は不十分だけど一応形になった
  （バージョン番号は修正等の日付とした）

#---------------------------------------------
#メインクラス
class ezffmpeg(Frame):
  def __init__(self,master=None,cnf={},**kw):
  def setnotebook(self):
  def removenotebook(self):
  def ondestfnchg(self):
  def createcmd(self):

#--------------------------------------------------------
# 実行ボタン、中止ボタン、コマンドコピーボタンのクラス
# duration:'00:00:00.00'形式
# createcmd:cmdリスト生成関数のアドレス
class runbuttons(Frame):
  def __init__(self,master=None,duration=None,createcmd=None,
               destfn=None,cnf={},**kw):
  def copycmd(self):
  def runclicked(self):
  def checkthread(self):
  def abortconv(self):
  def addprglabel(self):
  def updateprglabel(self,intval=0):
  def getstopflg(self):
  def runbtnon(self):

#--------------------------------------------------------
#映像編集をまとめるクラス
class editvideo(Frame):
  def __init__(self,master=None,
               srcmov=None,getdestfn=None,
               getsrcvideo=None,cnf={},**kw):
  def ondestfnchg(self):
  def sametimereducecrop(self,iam=None):
  def getargs(self):

#--------------------------------------------------------
# 映像コーデック変更（extcodecs.pyで主なコーデックと拡張子を列挙）
# srcmove:変換元辞書
# getdestfn:変換先パス名取得関数アドレス
class codecvideo(Frame):
  def __init__(self,master=None, srcmov=None, getdestfn=None, cnf={},**kw):
  def updatecmbbvalues(self):
  def getcodecs(self):
  def cbchanged(self,*args):
  def getvcodec(self):
  def getargs(self):

#---------------------------------------------------------
# 映像サイズの縮小（アスペクト比維持の選択）※cropと同時指定はできない
class reducevideo(Frame):
  def __init__(self,master=None,
               sametimechk=None,getsrcvideo=None,cnf={},**kw):
  def showhint(self,ev=None):
  def cbchanged(self,*args):
  def getargs(self):

#---------------------------------------------------------
# 映像領域の切り抜き　※reduceと同時指定はできない
class cropvideo(Frame):
  def __init__(self,master=None, getsrcvideo=None,
               sametimechk=None,cnf={},**kw):
  def showhint(self,ev=None):
  def checkxywh(self,ev):
  def cbchanged(self,*args):
  def getargs(self):

#--------------------------------------------------------
#音声編集をまとめるクラス
class editaudio(Frame):
  def __init__(self,master=None,srcmov=None,getdestfn=None,
               srcfn=None,cnf={},**kw):
  def ondestfnchg(self):
  def getargs(self):

#---------------------------------------------------------
#音声ボリュームの調整
class volumeaudio(Frame):
  def __init__(self,master=None,srcfn=None,cnf={},**kw):
  def cbchanged(self,*args):
  def getargs(self):

#----------------------------------------------------------
class codecaudio(codecvideo):  #codecvideoを継承（codecvideoの定義より後に）
  def __init__(self,master=None, srcmov=None, getdestfn=None, cnf={},**kw):
  def getcodecs(self):
  def getvcodec(self):
  def getargs(self):

#-------------------------------------------------------
#時間切り出し用のフレーム
#正確に切り出されるわけではないことに注意
#時間指定書式：00:00:00 (hh:mm:ss)
#（元々正確には切り出せないから秒未満を指定しても無意味）
class timecut(Frame):
  def __init__(self,master=None,duration='', cnf={},**kw):
  def cbchanged(self,*args):
  def gettimeformat(self):
  def spincheck(self):
  def getargs(self):

#--------------------------------------------------------
#video,audio,subtitleのストリームフレームをまとめたフレーム
#streamsdic:{'video':[{},],'audio':[{},], 'subtitle':[{},]}
class stream(Frame):
  def __init__(self,master=None,
               streamsdic={'video':[],'audio':[],'subtitle':[]},
               cnf={},**kw):
  def getargs(self):

#---------------------------------------------------------
#ストリーム選択クラスの雛形
#streamlabel: 表示用のラベル（映像、音声、字幕）
#streamlist:
#　srcfile.getsrcvideo,.getsrcaudio,.getsrcsubtitleの値をもとに表示
# [{'stream':'0:0','codec':'codec_str',..},{},..]
class selectstream(Frame):
  def __init__(self,master=None,typelabel='',streamlist=None,
               cnf={},**kw):
  def getcbtext(self,dic,wi=60,sei=''):
  def getmaplist(self):

#--------------------------------------------------
#出力先ファイルのクラス
class destfile(Frame):
  def __init__(self,master=None,onchange=None,cnf={},**kw):
  def saveas(self):
  def destfnchg(self,*args):
  def getdestfn(self):

#--------------------------------------------------
#変換元動画のクラス
#nbremove,nbset:外部関数のアドレス
#nbremove=notebook削除用の関数
#nbset=notebook配置用の関数
class srcfile(Frame):
  def __init__(self,master=None,nbremove=None,nbset=None,cnf={},**kw):
  def getduration(self):
  def getmovelement(self,*args):
  def getsrcvideo(self):
  def getsrcaudio(self):
  def getsrcsubtitle(self):
  def insertwrapstr(self,tx,wi=50,sei=''):
  def showsrcmov(self):
  def asksrcfile(self):
  def getsrcfn(self):

#-----------------------------------------------------
#ffmpegの[error]行をメッセージボックスに出力
def errmessage(errlist):

#---------------------------------------------
#元動画情報取得
#戻り値：辞書
def getsrcmov(fpath):

#----------------------------------------------
#処理メイン
def main():

'''
from tkinter import *
from tkinter import ttk
from tkinter import filedialog as fdlg
from tkinter import messagebox as mbx
from tkinter import simpledialog as smpldlg
import textwrap as twrap
import os,re,sys
from PIL import ImageTk,Image
import threading
from . import ffmp4cnv_inc as ffi
from . import scrollbar_inc as sci
from . import extcodecs as exc
from . import helpmessage_inc as hmi
from . import wait_tips_inc as wti

version = 'R7.5.4'
cropimage = 'cropping.png'
basecmd = 'ffmpeg -y -loglevel level+info'.split()
progress = '-progress -'.split()
#---------------------------------------------
#メインクラス
class ezffmpeg(Frame):
  def __init__(self,master=None,cnf={},**kw):
    self.master = master
    #Frameの__init__()呼び出し
    super().__init__(master,cnf,**kw)
    #クラス配置
    #srcfile
    self.sf = srcfile(self,nbremove=self.removenotebook,
                      nbset=self.setnotebook)
    self.sf.grid(row=0,column=0,sticky=W+E+N+S,padx=4)
    self.columnconfigure(0,weight=1)
    self.rowconfigure(0,weight=1)
    #destfile
    self.df = destfile(self,onchange=self.ondestfnchg)
    self.df.grid(row=1,column=0,sticky=W+E,padx=4,pady=2)

  #Notebookの表示は元動画の選択後に行う
  def setnotebook(self):
    #元のnotebookは削除する
    self.removenotebook()
    self.nb = ttk.Notebook(self)
    self.nb.grid(row=2,column=0,sticky=W+E)
    #選択中のtab色を変更
    s = ttk.Style()
    s.map("TNotebook.Tab", background= [('selected','yellow')])
    #notebookを親としてstream,timecut,video,audio,subtitleのframeを作成
    #stream
    self.stfr = stream(self.nb,
                  {'video':self.sf.getsrcvideo(),
                   'audio':self.sf.getsrcaudio(),
                   'subtitle':self.sf.getsrcsubtitle()})
    self.nb.add(self.stfr,text='ストリーム')
    #timecut
    self.tifr = timecut(self.nb,duration=self.sf.getduration())
    self.nb.add(self.tifr,text='時間切り出し')
    #editvideo
    self.vifr = editvideo(self.nb,
                          srcmov=self.sf.srcmov,
                          getdestfn=self.df.getdestfn,
                          getsrcvideo=self.sf.getsrcvideo)
    self.nb.add(self.vifr,text='映像',padding=4)
    #audio
    self.aufr = editaudio(self.nb,
                          srcmov=self.sf.srcmov,
                          getdestfn=self.df.getdestfn,
                          srcfn=self.sf.srcfn.get())
    self.nb.add(self.aufr,text='音声',padding=4)

    #実行ボタン関連
    self.runfr = runbuttons(self,duration=self.sf.getduration(),
                            createcmd=self.createcmd,
                            destfn=self.df.getdestfn())
    self.runfr.grid(row=3,column=0,sticky=W+E)

  #Notebookの削除
  def removenotebook(self):
    if 'nb' in dir(self):
      self.nb.destroy()
      mbx.showinfo('お知らせ','設定がクリアされました')

  #出力先ファイル名の変更に伴う処理
  def ondestfnchg(self):
    #有効な元動画が指定されていれば'self.nb'が存在するので
    if 'nb' in dir(self):
      #video 候補リストの変更
      self.vifr.ondestfnchg()
      #audio 候補リストの変更
      self.aufr.ondestfnchg()
      #実行ボタンを有効にする
      self.runfr.runbtnon()

  #cmdリスト生成
  def createcmd(self):
    #配下のフレームから現在の設定内容を収集してコマンドリストを生成
    cmdlist = []
    cmdlist.extend(basecmd)
    
    #元動画パス
    srcargs = []
    srcfn = self.sf.getsrcfn()
    srcargs.append('-i')
    srcargs.append(srcfn)
    #ストリーム
    mapargs = self.stfr.getargs()
    #時間切り出し
    timeargs  = self.tifr.getargs()
    #映像
    videoargs = self.vifr.getargs()
    #音声
    audioargs = self.aufr.getargs()
    #出力パス
    destfn = self.df.getdestfn()

    #コマンドリストを生成
    if timeargs:  #時間切り出し：開始時間
      cmdlist.extend(timeargs[:2])  #-ss xxx
    #元動画
    cmdlist.extend(srcargs)
    if timeargs:  #時間切り出し：長さ
      cmdlist.extend(timeargs[2:])  #-t xxx
    #ストリーム
    cmdlist.extend(mapargs)
    #映像:codec
    if not videoargs['reduce'] and not videoargs['crop']:
      cmdlist.extend(videoargs['codec'])
    else:  #reduce or crop がある場合
      if len(videoargs['codec'])==1 or videoargs['codec'][1] != 'copy':
        cmdlist.extend(videoargs['codec'])
    #映像:reduce
    if videoargs['reduce']:
      cmdlist.extend(videoargs['reduce'])
    #映像:crop
    if videoargs['crop']:
      cmdlist.extend(videoargs['crop'])
    #音声:codec
    if not audioargs['volume']:
      cmdlist.extend(audioargs['codec'])
    else: #volumeがある場合
      if len(audioargs['codec'])==1 or audioargs['codec'][1] != 'copy':
        cmdlist.extend(audioargs['codec'])
    #音声:volume
    if audioargs['volume']:
      cmdlist.extend(audioargs['volume'])
    #標準出力に進捗を出力 ※これがないと進捗が拾えない（忘れてた）
    cmdlist.extend(progress)
    #出力ファイル
    cmdlist.append(destfn)

    return cmdlist

#--------------------------------------------------------
# 実行ボタン、中止ボタン、コマンドコピーボタン
# duration:'00:00:00.00'形式
# createcmd:cmdリスト生成関数のアドレス
class runbuttons(Frame):
  def __init__(self,master=None,duration=None,createcmd=None,
               destfn=None,cnf={},**kw):
    self.master = master
    self.duration = duration
    self.createcmd = createcmd
    #Frameの__init__()呼び出し
    super().__init__(master,cnf,**kw)
    self.stopflg = False
    #中止ボタン(実行ボタンで配置される)
    self.abortbtntxt = StringVar()
    self.abortbtntxt.set('中止 (進捗{}%)'.format('0'))
    self.abortbtn = Button(self,textvariable=self.abortbtntxt,
                           command=self.abortconv)
    self.abortbtn.grid(row=0,column=0,sticky=W+E)
    self.abortbtn.grid_remove()
    #実行ボタン
    self.runbtn = Button(self,text='変換実行',command=self.runclicked,
                         width=30,state=NORMAL if destfn else DISABLED)
    self.runbtn.grid(row=0,column=0,sticky=W+E)
    #コマンドコピーボタン
    Button(self,text='実行コマンドをコピー',command=self.copycmd).grid(
      row=0,column=1,sticky=W+E)
    self.columnconfigure(0,weight=1)
    
  def copycmd(self):
    #現在の設定内容でffmpegの実行コマンドを生成してクリップボードに送る
    cmdlist = self.createcmd()
    #cmdlistをコマンドライン文字列に結合する
    cmdline = ''
    for i in cmdlist:
      if ' ' in i: #ファイル名などに空文字を含む場合あり
        cmdline += ' ' + '\'' + i + '\''
      else:
        cmdline += ' ' + i
    cmdline = cmdline[1:]  #先頭の空白除去
    self.clipboard_clear()
    self.clipboard_append(cmdline)
    mbx.showinfo('お知らせ','クリップボードにコマンドラインをコピーしました')

  #cmdlistに'-progress -'を指定するのを忘れて進捗が拾えずしばらく悩んだ
  def runclicked(self):
    #コマンドリスト生成
    cmdlist = self.createcmd()

    self.results = []
    #threadingでffi.doconv()を呼び出す
    ### def doconv(cmd,strfulltime,stopflg,pbdlg,results=[],showall=False) ##
    self.results = []
    self.thr = threading.Thread(target=ffi.doconv,
                  args=(cmdlist,self.duration,self,self,self.results))
    self.thr.start()

    #スレッドの状態を検知する
    self.after(200,self.checkthread)

  #.after()でスレッドの状況を200ms毎に調査
  def checkthread(self):
    if self.thr.is_alive():  #実行中
      self.after(200,self.checkthread)
    else:  #終了（エラー、中止を含む）
      #abortbtnを消してrunbtnを復活
      self.abortbtn.grid_remove()
      self.runbtn.grid()
      if self.results[0]:
        if not self.results[1]:
          mbx.showinfo('終了','変換が終了しました')
        else:
          mbx.showerror('エラー','\n'.join(self.results[1]))
      elif self.stopflg:  #中止ボタンが押されている場合
        mbx.showinfo('中止','中止しました')
      else:
        mbx.showinfo('エラー','不明なエラー')

  #中止ボタンが押された場合
  def abortconv(self):
    self.stopflg = True
    
  #変換実行ボタンを中止ボタンに置き換える
  def addprglabel(self):
    self.runbtn.grid_remove()
    self.abortbtn.grid()
  
  #進捗状況を中止ボタンに表示
  def updateprglabel(self,intval=0):
    self.abortbtntxt.set('中止 (進捗{}%)'.format(intval))
    self.abortbtn.update_idletasks()

  #中止ボタンの状況を返す
  def getstopflg(self):
    return self.stopflg

  #実行ボタンを有効にする
  def runbtnon(self):
    self.runbtn['state'] = NORMAL

#--------------------------------------------------------
#映像編集をまとめるクラス
class editvideo(Frame):
  def __init__(self,master=None,
               srcmov=None,getdestfn=None,
               getsrcvideo=None,cnf={},**kw):
    self.master = master
    #Frameの__init__()呼び出し
    super().__init__(master,cnf,**kw)
    #codec
    self.codecfr = codecvideo(self,srcmov=srcmov,
                              getdestfn=getdestfn)
    self.codecfr.grid(row=0,column=0,sticky=W+E)
    self.columnconfigure(0,weight=1)
    #reduce
    self.reducefr = reducevideo(self,getsrcvideo=getsrcvideo,
                                sametimechk=self.sametimereducecrop)
    self.reducefr.grid(row=1,column=0,sticky=W+E,pady=4)
    #crop
    self.cropfr = cropvideo(self,getsrcvideo=getsrcvideo,
                            sametimechk=self.sametimereducecrop)
    self.cropfr.grid(row=2,column=0,sticky=W+E)
  #親のondestfnchg()からコールされる
  #配下のフレーム内のメソッドを呼び出す
  def ondestfnchg(self):
    self.codecfr.updatecmbbvalues()
  #reduceとcropが同時にチェックされないよう調整する
  #iam:'reduce' or 'crop'　どちらのチェックON動作なのかを識別
  def sametimereducecrop(self,iam=None):
    if iam=='reduce' and self.cropfr.cb.get():
      self.cropfr.cb.set(0)
    elif iam=='crop' and self.reducefr.cb.get():
      self.reducefr.cb.set(0)
  #映像関係の引数辞書（リスト内蔵）を返す
  def getargs(self):
    videoargs = {}
    #codec
    videoargs['codec'] = self.codecfr.getargs()
    #reduce
    videoargs['reduce'] = self.reducefr.getargs()
    #crop
    videoargs['crop'] = self.cropfr.getargs()
    return videoargs  

#--------------------------------------------------------
# 映像コーデック変更（extcodecs.pyで主なコーデックと拡張子を列挙）
# srcmove:変換元辞書
# getdestfn:変換先パス名取得関数アドレス
class codecvideo(Frame):
  def __init__(self,master=None, srcmov=None, getdestfn=None, cnf={},**kw):
    self.master = master
    self.srcmov = srcmov
    self.getdestfn = getdestfn
    #Frameの__init__()呼び出し
    super().__init__(master,cnf,**kw)
    #checkbutton:コーデック変更
    self.cb = IntVar()
    self.cb.set(0)  #初期値はオフ
    Checkbutton(self,text='コーデック変更',takefocus=0,
                variable=self.cb).grid(row=0,column=0,sticky=W)
    #ttk.Combobox
    self.combo = StringVar()
    items = self.getcodecs()
    self.cmbb = ttk.Combobox(self,textvariable=self.combo,takefocus=0,
                             values=items,exportselection=0,
                             width=30,height=10)
    self.cmbb.grid(row=0,column=1)
    self.cmbb.state(['disabled',])
    #self.cbのtrace
    self.cb.trace('w',self.cbchanged)
    lt = '現在のコーデック：{}'.format(self.getvcodec())
    self.lb = Label(self,text=lt).grid(row=2,column=1,sticky=W)

  #出力先のパス名が変更された場合の選択肢の変更
  def updatecmbbvalues(self):
    self.cmbb['values'] = self.getcodecs()
  
  #出力先のコンテナで選択可能なvideoコーデックのリストを返す
  def getcodecs(self):
    fpath = self.getdestfn()
    if fpath:
      #出力先の拡張子を指定
      ext = os.path.splitext(fpath)[1]
      #拡張子なし又は該当なしの場合はelseと同じ
      codecs = exc.ext2codecs(ext)
    else: #出力先が未指定の場合（今後、このケースはなくする予定）
      codecs = exc.ext2codecs()
    return codecs['video']

  #チェックボタン連動
  #書き込み先が指定されていない場合は書き込み先を指定するよう
  #メッセージを表示するように変更
  def cbchanged(self,*args):
    dfpath = self.getdestfn()
    if dfpath:
      if self.cb.get():
        self.cmbb.state(['!disabled',])
      else:
        self.cmbb.state(['disabled',])
    else:  #書き込み先が未指定の場合
      #チェックボタンを強制的にクリア
      self.cb.set(0)
      #cmbbを強制的にdisabled
      self.cmbb.state(['disabled',])
      #メッセージ
      msg = '先に出力先の動画ファイル名を指定してください'
      mbx.showwarning('注意',msg,parent=self)
      
  #元動画のvcodec取得
  def getvcodec(self):
    #self.srcmovからvocdecを返す
    if 'video' in self.srcmov and self.srcmov['video'][0]:
      return self.srcmov['video'][0]['codec']
    else:
      return 'no video codec'

  #映像コーデック関係の引数リストを返す
  def getargs(self):
    if self.cb.get():
      codec = ['-vcodec',]
      combostr = self.combo.get()
      if not combostr:
        codec.append('copy')
      else:
        codec.extend(combostr.split())
    else:
      codec = ['-vcodec','copy']
    return codec
#---------------------------------------------------------
# 映像サイズの縮小（アスペクト比維持の選択）※cropと同時指定はできない
class reducevideo(Frame):
  def __init__(self,master=None,
               sametimechk=None,getsrcvideo=None,cnf={},**kw):
    self.master = master
    self.sametimechk = sametimechk
    #Frameの__init__()呼び出し
    super().__init__(master,cnf,**kw)
    #チェックボックスで縮小の有無を
    self.cb = IntVar()
    self.cb.set(0)  #初期値はオフ
    Checkbutton(self,text='映像サイズ（解像度）変更（書式 = x:y）',
                takefocus=0,variable=self.cb).grid(row=0,column=0,sticky=W)
    #選択肢取得
    vlist = getsrcvideo()
    if vlist and 'size' in vlist[0] and vlist[0]['size']:
      #ビデオが複数あっても先頭のビデオから
      downscales = ffi.getmovdownscale(vlist[0]['size'])
    else:
      downscales = []
    items = ['{}:{}'.format(i[0],i[1]) for i in downscales]
    #ドロップダウンリストで選択肢と自由選択も
    #ttk.Combobox
    self.combo = StringVar()
    self.cmbb = ttk.Combobox(self,textvariable=self.combo,takefocus=0,
                             values=items,exportselection=0,
                             width=12,height=10)
    self.cmbb.grid(row=0,column=1)
    self.cmbb.state(['disabled',])
    #help message
    hlp = Label(self,text='??(サイズ)',bg='pink',relief=GROOVE)
    hlp.grid(row=0,column=2,padx=10)
    hlp.bind('<Button-1>',self.showhint)
    #チェックボタンのtrace
    self.cb.trace('w',self.cbchanged)

  def showhint(self,ev=None):
    txt = 'サイズを直接指定する場合は横幅又は高さのいずれか\n一方を'
    txt += '指定して、他は -1 としてください。\n'
    txt += '例えば、\n横幅を800に指定する場合 800:-1\n'
    txt += '高さを500にする場合 -1:500'
    dlg = hmi.helpdialog(self,title='直接サイズを指定する場合',
                         text=txt,justify=LEFT,padx=4,pady=4)
    
  def cbchanged(self,*args):
    if self.cb.get():
      self.cmbb.state(['!disabled',])
      self.sametimechk(iam='reduce')
    else:
      self.cmbb.state(['disabled',])

  def getargs(self):
    if self.cb.get() and self.combo.get():
      cmd = '-vf scale=' + self.combo.get()
      return cmd.split()
    else:
      return []

#---------------------------------------------------------
# 映像領域の切り抜き
class cropvideo(Frame):
  def __init__(self,master=None, getsrcvideo=None,
               sametimechk=None,cnf={},**kw):
    self.master = master
    self.sametimechk = sametimechk
    #Frameの__init__()呼び出し
    super().__init__(master,cnf,**kw)
    #チェックボックスで映像領域切り抜きの有無を
    self.cb = IntVar()
    self.cb.set(0)  #初期値はオフ
    #先頭のVideo情報
    vdic = getsrcvideo()[0]
    size = vdic['size'] if 'size' in vdic and vdic['size'] else 'N/A'
    txt = '映像領域を切り抜き（元動画サイズ：{}）'.format(size)
    Checkbutton(self,text=txt,takefocus=0,
                variable=self.cb).grid(row=0,column=0,sticky=W)
    #元映像サイズ
    res = re.search('([0-9]*)x([0-9]*)',size)
    if res:
      self.size = {'x':int(res.group(1)),'y':int(res.group(2))}
    else:
      self.size = None
    #x:y:w:hの指定 entry? focus_outイベントで値チェック。focusを戻す？
    entryfr = Frame(self)
    entryfr.grid(row=1,column=0,sticky=W+E,padx=25)
    #x
    Label(entryfr,text='x:').grid(row=0,column=0)
    self.xval = StringVar()
    self.xval.set('0')
    self.xent = Entry(entryfr,textvariable=self.xval,state=DISABLED,
                 justify=CENTER,takefocus=0,width=5,highlightthickness=0)
    self.xent.grid(row=0,column=1,sticky=W)
    self.xent.bind('<FocusOut>',self.checkxywh)
    #y
    Label(entryfr,text=' y:').grid(row=0,column=2)
    self.yval = StringVar()
    self.yval.set('0')
    self.yent = Entry(entryfr,textvariable=self.yval,state=DISABLED,
                 justify=CENTER,takefocus=0,width=5,highlightthickness=0)
    self.yent.grid(row=0,column=3,sticky=W)
    self.yent.bind('<FocusOut>',self.checkxywh)
    #w
    Label(entryfr,text=' w:').grid(row=0,column=4)
    self.wval = StringVar()
    self.wval.set('0')
    self.went = Entry(entryfr,textvariable=self.wval,state=DISABLED,
                 justify=CENTER,takefocus=0,width=5,highlightthickness=0)
    self.went.grid(row=0,column=5,sticky=W)
    self.went.bind('<FocusOut>',self.checkxywh)
    #h
    Label(entryfr,text=' h:').grid(row=0,column=6)
    self.hval = StringVar()
    self.hval.set('0')
    self.hent = Entry(entryfr,textvariable=self.hval,state=DISABLED,
                 justify=CENTER,takefocus=0,width=5,highlightthickness=0)
    self.hent.grid(row=0,column=7,sticky=W)
    self.hent.bind('<FocusOut>',self.checkxywh)
    #xywhの参考表示ボタン
    b = Label(entryfr,text='??(x:y:w:h)',bg='pink',relief=GROOVE)
    b.grid(row=0,column=8,sticky=W,padx=20)
    b.bind('<Button-1>',self.showhint)
    #チェックボタンのtrace
    self.cb.trace('w',self.cbchanged)

  #hint callback
  def showhint(self,ev=None):
    #cropimage:モジュール変数
    imgpath = os.path.join(getscriptdir(),cropimage)
    img = Image.open(imgpath)
    tkimg = ImageTk.PhotoImage(img)
    dlg = hmi.helpdialog(self,title='領域切り抜き参考',
                         image=tkimg,buttontxt='close')

  #入力値のチェック
  def checkxywh(self,ev):
    if self.size:
      x = int(self.xval.get())
      y = int(self.yval.get())
      w = int(self.wval.get())
      h = int(self.hval.get())
      msg = ''
      #if x + w > self.size['x'] or y + h > self.size['y']:
      if x + w > self.size['x']:
        msg = 'x+wの値が{}を超えています'.format(self.size['x'])
      if y + h > self.size['y']:
        if msg:
          msg += '\n'
        msg += 'y+hの値が{}を超えています'.format(self.size['y'])
      if msg:
        mbx.showerror('入力値エラー',msg)
      
  #チェックボタンクリック時
  def cbchanged(self,*args):
    if self.cb.get():
      self.xent['state'] = NORMAL
      self.yent['state'] = NORMAL
      self.went['state'] = NORMAL
      self.hent['state'] = NORMAL
      if self.sametimechk:
        self.sametimechk(iam='crop')
    else:
      self.xent['state'] = DISABLED
      self.yent['state'] = DISABLED
      self.went['state'] = DISABLED
      self.hent['state'] = DISABLED

  def getargs(self):
    if self.cb.get():
      #-vf crop=w=960:h=540:x=100:y=100
      if self.wval.get() and self.hval.get(): #少なくともw,hに値必要
        cmdstr = '-vf crop=w={}:h={}:x={}:y={}'.format(self.wval.get(),
                                                       self.hval.get(),
                                                       self.xval.get(),
                                                       self.yval.get())
        cmd = cmdstr.split()
      else:
        msg = '切り出すサイズが指定されていないので無視します'
        mbx.showinfo('通知',msg)
        cmd = []
      return cmd
    else:
      return []
 
#--------------------------------------------------------
#音声編集をまとめる
class editaudio(Frame):
  def __init__(self,master=None,srcmov=None,getdestfn=None,
               srcfn=None,cnf={},**kw):
    self.master = master
    #Frameの__init__()呼び出し
    super().__init__(master,cnf,**kw)
    #codec
    self.codecfr = codecaudio(self,srcmov=srcmov,getdestfn=getdestfn)
    self.codecfr.grid(row=0,column=0,sticky=W+E)
    self.columnconfigure(0,weight=1)
    #volume & 推奨値を調査するボタン
    #srcfnを引数で与える
    self.volumefr = volumeaudio(self,srcfn)
    self.volumefr.grid(row=1,column=0,sticky=W+E)
  #親のondestfnchg()からコールされる
  #配下のフレーム内のメソッドを呼び出す
  def ondestfnchg(self):
    self.codecfr.updatecmbbvalues()

  def getargs(self):
    audiocmd = {}
    audiocmd['codec'] = self.codecfr.getargs()
    audiocmd['volume'] = self.volumefr.getargs()
    return audiocmd

#-----------------------------------------------------
#ffmpegの[error]行をメッセージボックスに出力
def errmessage(errlist):
  errmsg = ''
  for i in errlist:
    errmsg += i + '\n'
  errmsg = errmsg[:-1]
  mbx.showerror('エラー',errmsg)
  
#---------------------------------------------------------
#音声ボリュームの調整
class volumeaudio(Frame):
  def __init__(self,master=None,srcfn=None,cnf={},**kw):
    self.master = master
    #Frameの__init__()呼び出し
    super().__init__(master,cnf,**kw)
    self.cbval = IntVar()
    self.cbval.set(0)
    self.cb = Checkbutton(self,text='音声ボリューム補正',
                          variable=self.cbval,takefocus=0)
    self.cb.grid(row=0,column=0,sticky=W,columnspan=3)
    self.cbval.trace('w',self.cbchanged)
    #現在のmax_volume取得
    #breakpoint()
    #最大音量調査で時間がかかるので調査しない
##    maxvolstr,errlist = ffi.volumedetect(srcfn)
##    if len(errlist):
##      errmessage(errlist)

    #spinbox
    spnfr = Frame(self)
    self.vol = Spinbox(spnfr,from_=-20.0,to=20.0,width=6,
                       state=DISABLED,takefocus=0,increment=0.1,
                       justify=CENTER)
    self.vol.grid(row=0,column=0,sticky=W)
    Label(spnfr,text='dB').grid(row=0,column=1,sticky=W)
    spnfr.grid(row=1,column=0,sticky=W,padx=25)
    #初期値 ex. maxvolstr=='-3.0' inistr='3.0'
##    if maxvolstr:
##      maxvol = float(maxvolstr)
##      if maxvol < 0.0:
##        inivol = str(abs(maxvol))
##      else:
##        inivol = '0.0'
##    else:
##      inivol = '0.0'
    inivol = '0.0'
    self.vol['state'] = NORMAL
    self.vol.delete(0,END)
    self.vol.insert(0,inivol)
    self.vol['state'] = DISABLED
    #参考表示
    Label(spnfr,text='(現在の音量から増減する値)').grid(
      row=0,column=2)

  def cbchanged(self,*args):
    if self.cbval.get():
      self.vol['state'] = NORMAL
    else:
      self.vol['state'] = DISABLED

  def getargs(self):
    if self.cbval.get() and self.vol.get():
      cmdstr = '-af volume={}dB'.format(self.vol.get())
      return cmdstr.split()
    else:
      return []

#----------------------------------------------------------
class codecaudio(codecvideo):  #codecvideoを継承（codecvideoの定義より後に）
  def __init__(self,master=None, srcmov=None, getdestfn=None, cnf={},**kw):
    #codecvideoの__init__()呼び出し
    super().__init__(master,srcmov,getdestfn,cnf,**kw)

  #出力先のコンテナで選択可能なコーデックのリストを返す（audio用に変更）
  def getcodecs(self):
    fpath = self.getdestfn()
    if fpath:
      #出力先の拡張子を指定
      ext = os.path.splitext(fpath)[1]
      #拡張子なし又は該当なしの場合はelseと同じ
      codecs = exc.ext2codecs(ext)
    else: #出力先が未指定の場合（出力先未指定は想定外とする）
      codecs = exc.ext2codecs()
    return codecs['audio']

  #元動画のvcodec取得(audio用に変更)
  def getvcodec(self):
    #self.srcmovからaudiocdecを返す
    if 'audio' in self.srcmov and self.srcmov['audio'][0]:
      return self.srcmov['audio'][0]['codec']
    else:
      return 'no audio codec'

  def getargs(self):
    if self.cb.get():
      codec = self.combo.get()
      if not codec.strip():
        codec = ['-acodec','copy']
      else:
        codec = ['-acodec',codec.strip()]
    else:
      codec = ['-acodec','copy']
    return codec

#-------------------------------------------------------
#時間切り出し用のフレーム
#正確に切り出されるわけではないことに注意
#時間指定書式：00:00:00 (hh:mm:ss)　
#（元々正確には切り出せないから秒未満を指定しても無意味）　
class timecut(Frame):
  def __init__(self,master=None,duration='', cnf={},**kw):
    self.master = master
    self.duration = duration
    self.maxsec = ffi.strtimetosec(duration)
    #Frameの__init__()呼び出し
    super().__init__(master,cnf,**kw)
    #
    self.cbval = IntVar()
    self.cbval.set(0)
    self.cb = Checkbutton(self,text='時間切り出し：最大({})'.format(duration),
                          variable=self.cbval,takefocus=0)
    self.cb.grid(row=0,column=0,sticky=W,columnspan=2,padx=10)
    self.cbval.trace('w',self.cbchanged)
    #start time
    stfr = Frame(self)
    Label(stfr,text='開始時間').grid(row=0,column=0,columnspan=6,sticky=W)
    self.sth = Spinbox(stfr,from_=0,to=24,width=2,
                       command=self.spincheck,state=DISABLED)
    self.sth.grid(row=1,column=0)
    Label(stfr,text='h ').grid(row=1,column=1)
    self.stm = Spinbox(stfr,from_=0,to=59,width=2,
                       command=self.spincheck,state=DISABLED)
    self.stm.grid(row=1,column=2)
    Label(stfr,text='m ').grid(row=1,column=3)
    self.sts = Spinbox(stfr,from_=0,to=59,width=2,
                       command=self.spincheck,state=DISABLED)
    self.sts.grid(row=1,column=4)
    Label(stfr,text='s').grid(row=1,column=5)
    stfr.grid(row=1,column=0,sticky=W,padx=30)
    #time lengs
    lenfr = Frame(self)
    Label(lenfr,text='切り出す長さ').grid(row=0,column=0,columnspan=6,sticky=W)
    self.lenh = Spinbox(lenfr,from_=0,to=24,width=2,
                        command=self.spincheck,state=DISABLED)
    self.lenh.grid(row=1,column=0)
    Label(lenfr,text='h ').grid(row=1,column=1)
    self.lenm = Spinbox(lenfr,from_=0,to=59,width=2,
                        command=self.spincheck,state=DISABLED)
    self.lenm.grid(row=1,column=2)
    Label(lenfr,text='m ').grid(row=1,column=3)
    self.lens = Spinbox(lenfr,from_=0,to=59,width=2,
                        command=self.spincheck,state=DISABLED)
    self.lens.grid(row=1,column=4)
    Label(lenfr,text='s').grid(row=1,column=5)
    lenfr.grid(row=1,column=1,sticky=W,padx=20)

  def cbchanged(self,*args):
    if self.cbval.get():
      self.sth['state'] = 'readonly'
      self.stm['state'] = 'readonly'
      self.sts['state'] = 'readonly'
      self.lenh['state'] = 'readonly'
      self.lenm['state'] = 'readonly'
      self.lens['state'] = 'readonly'
    else:
      self.sth['state'] = DISABLED
      self.stm['state'] = DISABLED
      self.sts['state'] = DISABLED
      self.lenh['state'] = DISABLED
      self.lenm['state'] = DISABLED
      self.lens['state'] = DISABLED

  #開始時間と切り出す長さの時間文字列を返す
  #'00:00:00'形式
  def gettimeformat(self):
    ststr = '{}:{}:{}'.format(self.sth.get(),self.stm.get(),self.sts.get())
    lenstr = '{}:{}:{}'.format(self.lenh.get(),self.lenm.get(),
                               self.lens.get())
    return [ststr,lenstr]
  
  #スピンボックスの値をチェック
  def spincheck(self):
    #self.maxsec 最大秒数
    #開始時間秒と切り出し時間秒を足して最大秒数以内
    ststr,lenstr = self.gettimeformat()
    if ffi.strtimetosec(ststr)+ffi.strtimetosec(lenstr) > self.maxsec:
      #オーバータイム
      mbx.showerror('警告','切り出し範囲がオーバーしています。')

  #時間切り出しの引数リストを返す
  def getargs(self):
    if self.cbval.get():
      ststr,lenstr = self.gettimeformat()
      stsec = ffi.strtimetosec(ststr)
      lensec = ffi.strtimetosec(lenstr)
      if lensec > 0:
        args = []
        args.append('-ss')
        args.append('{}'.format(stsec))
        args.append('-t')
        args.append('{}'.format(lensec))
        return args
      else:
        return []
    else:
      return []
#--------------------------------------------------------
#video,audio,subtitleのストリームフレームをまとめたフレーム
#streamsdic:{'video':[{},],'audio':[{},], 'subtitle':[{},]}
class stream(Frame):
  def __init__(self,master=None,
               streamsdic={'video':[],'audio':[],'subtitle':[]},
               cnf={},**kw):
    self.master = master
    #Frameの__init__()呼び出し
    super().__init__(master,cnf,**kw)
    #video
    self.vifr = selectstream(self,typelabel='映像：',
                              streamlist=streamsdic['video'])
    #audio
    self.aufr = selectstream(self,typelabel='音声：',
                              streamlist=streamsdic['audio'])
    #subtitle
    self.sufr = selectstream(self,typelabel='字幕：',
                              streamlist=streamsdic['subtitle'])
    self.vifr.grid(row=0,column=0,sticky=W+E)
    self.aufr.grid(row=1,column=0,sticky=W+E)
    self.sufr.grid(row=2,column=0,sticky=W+E)
    self.columnconfigure(0,weight=1)

  #ffmpeg用のmap引数を返す
  def getargs(self):
    stv = self.vifr.getmaplist()
    sta = self.aufr.getmaplist()
    sts = self.sufr.getmaplist()
    stv.extend(sta)
    stv.extend(sts)
    return stv

#---------------------------------------------------------
#ストリーム選択クラスの雛形
#streamlabel: 表示用のラベル（映像、音声、字幕）
#streamlist:
#　srcfile.getsrcvideo,.getsrcaudio,.getsrcsubtitleの値をもとに表示
# [{'stream':'0:0','codec':'codec_str',..},{},..]
class selectstream(Frame):
  def __init__(self,master=None,typelabel='',streamlist=None,
               cnf={},**kw):
    self.master = master
    #Frameの__init__()呼び出し
    super().__init__(master,cnf,**kw)
    #項目
    # streamlist:の数だけLabelのspanが必要
    streamcnt = len(streamlist)
    streamcnt = 1 if not streamcnt else streamcnt
    Label(self,text=typelabel).grid(row=0,column=0,sticky=W+N,
                                    rowspan=streamcnt)
    
    #最終的に☑されているマップ文字列をリストで返すことが目的なので
    #onvalueはマップ文字列とする
    self.varlist = []
    for dic in streamlist:
      self.varlist.append(StringVar())
      linenum = len(self.varlist)-1
      var = self.varlist[linenum]
      #?? wraplengthオプションの動作が変なので使えない
      ck = Checkbutton(self,variable=var,onvalue=dic['stream'],offvalue='',
                  text=self.getcbtext(dic),takefocus=0)
      ck.grid(row=linenum,column=1,sticky=W+N)
      ck.select()
    
  #辞書の内容を整形テキストにして返す
  #checkbutonのwraplengthオプションの動作が変なのでtextwrap.wrapを使う
  #dic:辞書, wi:最大テキスト長, sei:2行目のインデント文字列
  def getcbtext(self,dic,wi=60,sei=''):
    cbtxt = ''
    for k in dic:
      cbtxt += '{}:{}, '.format(k,dic[k])
    if cbtxt:
      cbtxt = cbtxt[:-2]
    res = twrap.wrap(cbtxt,width=wi,subsequent_indent=sei)
    res = '\n'.join(res)
    return res

  #選択されているマップ文字列のリストを返す
  #戻り値：['-map','o:o','-map','0:1',..]
  def getmaplist(self):
    stlist = []
    for v in self.varlist:
      st = v.get()
      if st:
        stlist.append('-map')
        stlist.append(st)
    return stlist

#--------------------------------------------------
#出力先ファイルのクラス
class destfile(Frame):
  def __init__(self,master=None,onchange=None,cnf={},**kw):
    self.master = master
    self.onchange = onchange
    super().__init__(master,cnf,**kw)
    Label(self,text='出力先：').grid(row=0,column=0,sticky=W)
    self.destfn = StringVar()
    #entry入力1字毎にイベントが発生するのでreadonlyに
    self.destent = Entry(self,textvariable=self.destfn,state='readonly')
    self.destent.grid(row=0,column=1,sticky=W+E)
    Button(self,text='参照',command=self.saveas).grid(row=0,column=2)
    self.columnconfigure(0,weight=0)
    self.columnconfigure(1,weight=1)
    self.columnconfigure(2,weight=0)
    self.destfn.trace('w',self.destfnchg)

  def saveas(self):
    res = fdlg.asksaveasfilename(title='出力先のファイル名を指定')
    if res:
      self.destfn.set(res)

  #変更された場合、親フレーム側で必要な処理をさせる
  def destfnchg(self,*args):
    #拡張子に応じたコーデックの選択肢の設定と選択済みコーデックの検査
    #print('class destfile.destfnchg()',self.destfn.get())
    self.onchange()

  #現在の出力先のパス名を返す
  def getdestfn(self):
    return self.destfn.get()
  
#--------------------------------------------------
#変換元動画のクラス
#nbremove,nbset:外部関数のアドレス
#nbremove=notebook削除用の関数
#nbset=notebook配置用の関数
class srcfile(Frame):
  def __init__(self,master=None,nbremove=None,nbset=None,cnf={},**kw):
    self.master = master
    self.nbremove = nbremove
    self.nbset = nbset
    self.srcmov = {}
    #Frameの__init__()呼び出し
    super().__init__(master,cnf,**kw)
    Label(self,text='変換元：').grid(row=0,column=0,sticky=W)
    self.srcfn = StringVar()
    self.srcent = Entry(self,textvariable=self.srcfn,
          state='readonly')
    self.srcent.grid(row=0,column=1,sticky=W+E)
    Button(self,text='参照',
           command=self.asksrcfile).grid(row=0,column=2)
    self.columnconfigure(0,weight=0)
    self.columnconfigure(1,weight=1)
    self.columnconfigure(2,weight=0)
    #trace
    self.srcfn.trace('w',self.getmovelement)
    #text-widget
    self.txfr = LabelFrame(self,text='変換元動画の概要')
    self.txw = Text(self.txfr,height=8)
    sci.scrollbarset(self.txfr,self.txw)
    self.txfr.grid(row=1,column=0,columnspan=3,sticky=W+E)
    self.txw['state'] = DISABLED
    
  #動画の時間長を返す
  def getduration(self):
    if 'duration' in self.srcmov:
      return self.srcmov['duration']
    else:
      return ''
    
  #self.srcfnの値変更に対するtrace callback
  def getmovelement(self,*args):
    #tips on
    msg = '変換元動画を調べています...'
    tips = wti.wait_tips(self,message=msg,txw=30,txh=3)
    tips.update_idletasks()
    s = {}
    s['duration'],s['vbitrate'],s['video'],s['audio'],\
      s['subtitle'] = ffi.getffprobe(['ffprobe',self.srcfn.get()])
    if not s['video']:
      self.srcmov = {}
      msg = '指定されたファイルが存在しないか動画ファイルではありません'
      mbx.showerror('error',msg,parent=self.master)
      self.srcent['fg'] = 'red'
      self.txw['state'] = NORMAL
      self.txw.delete('0.0',END)
      self.txw['state'] = DISABLED
      #Notebookが表示されていたら消す又は削除
      #self.master.removenotebook()
      self.nbremove()
    else:
      self.srcmov = s
      self.srcent['fg'] = 'black'
      #情報表示へ
      self.showsrcmov()
      #Notebookウィジェットを作成して各種変換オプションを表示
      #self.master.setnotebook()
      self.nbset()
    #wait_tips off
    tips.cancel()
    tips = None

  #元動画のビデオ情報（dics in list）
  #戻り値：[{stream:,codec:,size:},..]
  def getsrcvideo(self):
    if not self.srcmov:
      return []
    vlist = []
    for mdic in self.srcmov['video']:
      vdic = {}
      vdic['stream'] = mdic['stream']
      vdic['lang'] = mdic['lang']
      vdic['codec'] = mdic['codec']
      vdic['size'] = mdic['size']
      if not vdic['size'] and 'line' in mdic:
        #lineから取得を試みる
        res = re.search(' ([0-9]*?)x([0-9]*?) ',mdic['line'])
        if res and res.group(0):
          vdic['size'] = '{}x{}'.format(res.group(1),res.group(2))
      vlist.append(vdic)
    return vlist

  #元動画の音声情報（dics in list）
  #[{stream:,lang:,codec:},..]
  def getsrcaudio(self):
    if not self.srcmov:
      return []
    #stream,codecは取得できているはず
    alist = []
    for adic in self.srcmov['audio']:
      bdic = {}
      bdic['stream'] = adic['stream']
      bdic['lang'] = adic['lang']
      bdic['codec'] = adic['codec']
      if not bdic['lang'] and 'line' in adic:
        #lineから取得を試みる
        res = re.search('Stream #[0-9]?:[0-9]?(.*?): Audio:',adic['line'])
        if res and res.group(0):
          bdic['lang'] = res.group(1)
      alist.append(bdic)
    return alist

  #元動画の字幕情報 (dics in list)
  #[{stream:,lang:},..]
  def getsrcsubtitle(self):
    if not self.srcmov:
      return []
    sblist = []
    for sub in self.srcmov['subtitle']:
      sdic = {}
      sdic['stream'] = sub['stream']
      sdic['lang'] = sub['lang']
      sblist.append(sdic)
    return sblist
  
  #showsrcmovのサポート関数
  #与えられた文字列を整形してテキストウィジェットに挿入する
  #tx:テキスト,wi:制限文字列長,sei:２行目以降のインデント文字列
  def insertwrapstr(self,tx,wi=50,sei=''):
    lines = twrap.wrap(tx,width=wi,subsequent_indent=sei)
    for li in lines:
      self.txw.insert(END,li+'\n')

  #srcmovの概要を表示
  def showsrcmov(self):
    self.txw['state'] = NORMAL
    self.txw.delete('0.0',END)
    for item in self.srcmov:
      if item=='duration':
        self.txw.insert(END,'時間長：{}\n'.format(self.srcmov[item]))
      elif item=='vbitrate':
        self.txw.insert(END,'映像ﾋﾞｯﾄﾚｰﾄ：{}kb/s\n'.format(self.srcmov[item]))
      elif item=='video': #dics in list
        if self.srcmov[item]:
          #textwrapで整形する必要がある
          #１行の文字数を制限しカンマの位置で折り返し、２行目以降にインデントを入れる
          for i,dic in enumerate(self.srcmov[item]):
            if 'line' in dic:
              self.insertwrapstr('映像{}の参考：{}'.format(i+1,dic['line']),
                                 wi=50,sei=' '*6)
            else:
              msg = '映像{}：Stream：{}, 言語：{}, codec：{}, 映像サイズ：{}'
              msg = msg.format(i+1,dic['stream'],dic['lang'],dic['codec'],
                               dic['size'])
              self.insertwrapstr(msg,wi=50,sei=' '*6)
      elif item=='audio':
        if self.srcmov[item]:
          for i,dic in enumerate(self.srcmov[item]):
            if 'line' in dic:
              self.insertwrapstr('音声{}の参考：{}'.format(i+1,dic['line']),
                                  wi=50,sei=' '*6)
            else:
              msg = '音声{}：Stream：{}, 言語：{}, codec：{}, ｻﾝﾌﾟﾘﾝｸﾞﾚｰﾄ：{}Hz, '
              msg += '音源：{}, 音声ﾋﾞｯﾄﾚｰﾄ：{}kb/s'
              msg = msg.format(i+1,dic['stream'],dic['lang'],dic['codec'],
                               dic['samprate'],dic['stereo'],dic['abitrate'])
              self.insertwrapstr(msg,wi=50,sei=' '*6)
      elif item=='subtitle':
        if self.srcmov[item]:
          for i,dic in enumerate(self.srcmov[item]):
            msg = '字幕{}：Stream：{}, 言語：{}\n'.format(i+1,dic['stream'],
                                                   dic['lang'])
            self.txw.insert(END,msg)
    self.txw['state'] = DISABLED  

  #ファイルオープンダイアログ      
  def asksrcfile(self):
    res = fdlg.askopenfilename(title='変換元動画の指定')
    if res:
      self.srcfn.set(res)
  #ソースファイルパスを返す
  def getsrcfn(self):
    return self.srcfn.get()
#---------------------------------------------
#元動画情報取得
#戻り値：辞書
def getsrcmov(fpath):
  srcmov = {}
  srcmov['duration'],srcmov['vbitrate'],srcmov['video'],\
    srcmov['audio'],srcmov['subtitle'] = ffi.ffprobe(['ffprobe',fpath])
  return srcmov  
#-----------------------------------------------
###ウィンドウの表示位置を調整
## ??? 現在のウィンドウの高さがうまく取得できないから保留
##def locatewindow():
##  global root
##  #root.geometry()で調整する
##  #現在のウィンドウサイズと位置情報
##  root.update_idletasks()
##  geostr = root.winfo_geometry()
##  w = root.winfo_reqwidth()
##  h = root.winfo_reqheight()
##  #スクリーンサイズ
##  screenw = root.winfo_screenwidth()
##  screenh = root.winfo_screenheight()

#-----------------------------------------
#カレントディレクトリを返す
def getscriptdir():
  if getattr(sys,'frozen',False):
    #sysオブジェクトにfrozenアトリビュートが存在すればその値が
    #frozenアトリビュートが存在しなければFalseが返る
    #Trueならfrozen環境
    return os.path.dirname(sys.executable)
  else:
    #frozen環境ではない。
    return os.path.dirname(os.path.realpath(__file__))
  
#----------------------------------------------
#処理メイン
def main():
  #global root
  root = Tk()
  #root.geometry('500x500')  #すべてのクラスを配置してから調整
  #無駄な余白が生じないように手動でのサイズ変更は禁止
  root.resizable(width=False,height=False)
  root.title('ezffmpeg ({})'.format(version))
  if ffi.checkffmpeg(root):
    ezf = ezffmpeg(root)
    ezf.grid(row=0,column=0,padx=4,pady=4,sticky=N+S+W+E)
    root.columnconfigure(0,weight=1)
    #root.rowconfigure(0,weight=1)  #全クラスが配置されてから
    root.update_idletasks()
    root.mainloop()
  else:
    root.destry()

#----------------------------------------------------  
if __name__=='__main__':
  main()
  
